<?php
/**
 * QuickBooks Web Connector bridge (QBWC) – WSDL mode, PHP 7+
 * - Serves WSDL at ?wsdl
 * - Health check at ?health=1
 * - QBWC SOAP methods (serverVersion, clientVersion, authenticate, sendRequestXML, receiveResponseXML, getLastError, closeConnection, connectionError)
 * - Minimal queue driver using table `qbwc_queue`
 *
 * Safe defaults: no fatal output; all diagnostics to qbwc_server2.log
 */

declare(strict_types=1);
error_reporting(E_ALL);
ini_set('display_errors', '0');

/*───────────────────────────────────────────────────────────────────────────*/
/* CONFIG                                                                   */
/*───────────────────────────────────────────────────────────────────────────*/
const BRIDGE_VERSION = '2.3.4';          // shown to QBWC & health
const LOG_FILE       = __DIR__ . '/qbwc_server2.log';

const DB_HOST    = 'localhost';
const DB_NAME    = 'rkimport_lat4';
const DB_USER    = 'rkimport_qbws';
const DB_PASS    = '0MWD4.@HgG5F;Dr[';
const DB_CHARSET = 'utf8mb4';

const QUEUE_TABLE = 'qbwc_queue';        // table name we’ll use

// Leave blank string to use the currently-open company file in QuickBooks
const QB_COMPANY_FILE = '';

/* Optional app creds (we do NOT hard-fail if they differ; QBWC already gates access) */
const APP_USERNAME = 'qbwc_user';
const APP_PASSWORD = 'test1234';

/*───────────────────────────────────────────────────────────────────────────*/
/* Helpers                                                                  */
/*───────────────────────────────────────────────────────────────────────────*/
function log_line(string $msg): void {
    static $pid;
    if ($pid === null) { $pid = getmypid(); }
    @file_put_contents(
        LOG_FILE,
        '[' . gmdate('Y-m-d H:i:s') . "Z][$pid] $msg\n",
        FILE_APPEND
    );
}

function pdo(): PDO {
    static $pdo = null;
    if ($pdo) return $pdo;
    $dsn = 'mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=' . DB_CHARSET;
    $pdo = new PDO($dsn, DB_USER, DB_PASS, [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES   => false,
    ]);
    return $pdo;
}

function current_endpoint(): string {
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host   = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $path   = strtok($_SERVER['REQUEST_URI'] ?? $_SERVER['SCRIPT_NAME'], '?');
    return "$scheme://$host$path";
}

/*───────────────────────────────────────────────────────────────────────────*/
/* WSDL (inline)                                                            */
/*───────────────────────────────────────────────────────────────────────────*/
function qbwc_wsdl(): string {
    $endpoint = htmlspecialchars(current_endpoint(), ENT_QUOTES, 'UTF-8');
    // Standard, document/literal WSDL the QBWC samples expect
    return <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<definitions xmlns:wsdl="http://schemas.xmlsoap.org/wsdl/"
             xmlns:soap="http://schemas.xmlsoap.org/wsdl/soap/"
             xmlns:xsd="http://www.w3.org/2001/XMLSchema"
             xmlns:tns="http://developer.intuit.com/"
             targetNamespace="http://developer.intuit.com/">
  <types>
    <xsd:schema targetNamespace="http://developer.intuit.com/">
      <xsd:complexType name="ArrayOfString">
        <xsd:sequence>
          <xsd_element name="string" type="xsd:string" minOccurs="0" maxOccurs="unbounded"/>
        </xsd:sequence>
      </xsd:complexType>
    </xsd:schema>
  </types>

  <!-- Messages -->
  <message name="serverVersionRequest"/>
  <message name="serverVersionResponse"><part name="serverVersionResult" type="xsd:string"/></message>

  <message name="clientVersionRequest"><part name="strVersion" type="xsd:string"/></message>
  <message name="clientVersionResponse"><part name="clientVersionResult" type="xsd:string"/></message>

  <message name="authenticateRequest">
    <part name="strUserName" type="xsd:string"/>
    <part name="strPassword" type="xsd:string"/>
  </message>
  <message name="authenticateResponse"><part name="authenticateResult" type="tns:ArrayOfString"/></message>

  <message name="sendRequestXMLRequest">
    <part name="ticket" type="xsd:string"/>
    <part name="strHCPResponse" type="xsd:string"/>
    <part name="strCompanyFileName" type="xsd:string"/>
    <part name="qbXMLCountry" type="xsd:string"/>
    <part name="qbXMLMajorVers" type="xsd:int"/>
    <part name="qbXMLMinorVers" type="xsd:int"/>
  </message>
  <message name="sendRequestXMLResponse"><part name="sendRequestXMLResult" type="xsd:string"/></message>

  <message name="receiveResponseXMLRequest">
    <part name="ticket" type="xsd:string"/>
    <part name="response" type="xsd:string"/>
    <part name="hresult" type="xsd:string"/>
    <part name="message" type="xsd:string"/>
  </message>
  <message name="receiveResponseXMLResponse"><part name="receiveResponseXMLResult" type="xsd:int"/></message>

  <message name="getLastErrorRequest"><part name="ticket" type="xsd:string"/></message>
  <message name="getLastErrorResponse"><part name="getLastErrorResult" type="xsd:string"/></message>

  <message name="connectionErrorRequest">
    <part name="ticket" type="xsd:string"/>
    <part name="hresult" type="xsd:string"/>
    <part name="message" type="xsd:string"/>
  </message>
  <message name="connectionErrorResponse"><part name="connectionErrorResult" type="xsd:string"/></message>

  <message name="closeConnectionRequest"><part name="ticket" type="xsd:string"/></message>
  <message name="closeConnectionResponse"><part name="closeConnectionResult" type="xsd:string"/></message>

  <!-- PortType -->
  <portType name="QBWebConnectorSvcSoap">
    <operation name="serverVersion">
      <input message="tns:serverVersionRequest"/>
      <output message="tns:serverVersionResponse"/>
    </operation>
    <operation name="clientVersion">
      <input message="tns:clientVersionRequest"/>
      <output message="tns:clientVersionResponse"/>
    </operation>
    <operation name="authenticate">
      <input message="tns:authenticateRequest"/>
      <output message="tns:authenticateResponse"/>
    </operation>
    <operation name="sendRequestXML">
      <input message="tns:sendRequestXMLRequest"/>
      <output message="tns:sendRequestXMLResponse"/>
    </operation>
    <operation name="receiveResponseXML">
      <input message="tns:receiveResponseXMLRequest"/>
      <output message="tns:receiveResponseXMLResponse"/>
    </operation>
    <operation name="getLastError">
      <input message="tns:getLastErrorRequest"/>
      <output message="tns:getLastErrorResponse"/>
    </operation>
    <operation name="connectionError">
      <input message="tns:connectionErrorRequest"/>
      <output message="tns:connectionErrorResponse"/>
    </operation>
    <operation name="closeConnection">
      <input message="tns:closeConnectionRequest"/>
      <output message="tns:closeConnectionResponse"/>
    </operation>
  </portType>

  <!-- Binding -->
  <binding name="QBWebConnectorSvcSoap" type="tns:QBWebConnectorSvcSoap">
    <soap:binding style="document" transport="http://schemas.xmlsoap.org/soap/http"/>
    <operation name="serverVersion">
      <soap:operation soapAction="http://developer.intuit.com/serverVersion"/>
      <input><soap:body use="literal"/></input>
      <output><soap:body use="literal"/></output>
    </operation>
    <operation name="clientVersion">
      <soap:operation soapAction="http://developer.intuit.com/clientVersion"/>
      <input><soap:body use="literal"/></input>
      <output><soap:body use="literal"/></output>
    </operation>
    <operation name="authenticate">
      <soap:operation soapAction="http://developer.intuit.com/authenticate"/>
      <input><soap:body use="literal"/></input>
      <output><soap:body use="literal"/></output>
    </operation>
    <operation name="sendRequestXML">
      <soap:operation soapAction="http://developer.intuit.com/sendRequestXML"/>
      <input><soap:body use="literal"/></input>
      <output><soap:body use="literal"/></output>
    </operation>
    <operation name="receiveResponseXML">
      <soap:operation soapAction="http://developer.intuit.com/receiveResponseXML"/>
      <input><soap:body use="literal"/></input>
      <output><soap:body use="literal"/></output>
    </operation>
    <operation name="getLastError">
      <soap:operation soapAction="http://developer.intuit.com/getLastError"/>
      <input><soap:body use="literal"/></input>
      <output><soap:body use="literal"/></output>
    </operation>
    <operation name="connectionError">
      <soap:operation soapAction="http://developer.intuit.com/connectionError"/>
      <input><soap:body use="literal"/></input>
      <output><soap:body use="literal"/></output>
    </operation>
    <operation name="closeConnection">
      <soap:operation soapAction="http://developer.intuit.com/closeConnection"/>
      <input><soap:body use="literal"/></input>
      <output><soap:body use="literal"/></output>
    </operation>
  </binding>

  <!-- Service -->
  <service name="QBWebConnectorSvc">
    <port name="QBWebConnectorSvcSoap" binding="tns:QBWebConnectorSvcSoap">
      <soap:address location="{$endpoint}"/>
    </port>
  </service>
</definitions>
XML;
}

/*───────────────────────────────────────────────────────────────────────────*/
/* Health                                                                    */
/*───────────────────────────────────────────────────────────────────────────*/
if (isset($_GET['health'])) {
    header('Content-Type: application/json; charset=utf-8');
    $rows = null;
    try {
        $rows = (int) pdo()->query("SELECT COUNT(*) FROM " . QUEUE_TABLE)->fetchColumn();
    } catch (Throwable $e) {
        $rows = -1;
    }
    echo json_encode([
        'ok'         => true,
        'version'    => BRIDGE_VERSION,
        'queue_table'=> QUEUE_TABLE,
        'queue_rows' => $rows
    ]);
    exit;
}

/*───────────────────────────────────────────────────────────────────────────*/
/* WSDL delivery                                                             */
/*───────────────────────────────────────────────────────────────────────────*/
if (isset($_GET['wsdl'])) {
    header('Content-Type: text/xml; charset=utf-8');
    echo qbwc_wsdl();
    exit;
}

/*───────────────────────────────────────────────────────────────────────────*/
/* QBWC Service                                                              */
/*───────────────────────────────────────────────────────────────────────────*/
class QBWCService
{
    public function serverVersion(): string {
        log_line("serverVersion() -> " . BRIDGE_VERSION);
        return BRIDGE_VERSION;
    }

    public function clientVersion(string $strVersion): string {
        log_line("clientVersion($strVersion) -> ok");
        // return a warning string to force updates; empty string = ok
        return '';
    }

    public function authenticate(string $strUserName, string $strPassword): array {
        log_line("authenticate(user=$strUserName)");
        // If you want to enforce creds, uncomment the check below.
        // if ($strUserName !== APP_USERNAME || $strPassword !== APP_PASSWORD) {
        //     log_line("authenticate -> invalid credentials");
        //     return ["nvu", "", "", ""]; // not valid user
        // }

        $token = 'TOKEN' . substr(bin2hex(md5(uniqid('', true), true)), 0, 12);
        $company = QB_COMPANY_FILE; // "" => currently open file
        $ret = [$token, $company, "", ""];
        log_line("authenticate -> ticket=$token, cfn=" . ($company === '' ? '<current>' : $company));
        return $ret;  // *** Indexed array of 4 strings (what QBWC expects) ***
    }

    public function sendRequestXML(
        string $ticket,
        string $strHCPResponse,
        string $strCompanyFileName,
        string $qbXMLCountry,
        int $qbXMLMajorVers,
        int $qbXMLMinorVers
    ): string {
        // Try to claim the next queued job and return its qbXML
        try {
            $pdo = pdo();
            $pdo->beginTransaction();

            // Lock one row
            $row = $pdo->query("
                SELECT * FROM " . QUEUE_TABLE . "
                 WHERE status IN ('queued','retry')
                   AND qbxml_sent = 0
                   AND attempts < 5
                 ORDER BY id ASC
                 LIMIT 1
                 FOR UPDATE
            ")->fetch();

            if (!$row) {
                $pdo->commit();
                log_line("sendRequestXML(ticket=$ticket) -> (no work)");
                return ''; // no work, QBWC will call closeConnection
            }

            // mark as in-flight & increment attempts
            $stmt = $pdo->prepare("
                UPDATE " . QUEUE_TABLE . "
                   SET status='processing',
                       qbxml_sent=1,
                       qbxml_sent_at=NOW(),
                       attempts=attempts+1,
                       updated_at=NOW()
                 WHERE id=:id AND qbxml_sent=0
            ");
            $stmt->execute([':id' => $row['id']]);
            $pdo->commit();

            $xml = (string)($row['qbxml_request'] ?? '');
            log_line("sendRequestXML -> id={$row['id']}, bytes=" . strlen($xml));
            return $xml; // If empty, QBWC will effectively do nothing this cycle

        } catch (Throwable $e) {
            try { pdo()->rollBack(); } catch (Throwable $e2) {}
            log_line("sendRequestXML error: " . $e->getMessage());
            return ''; // safest: do no work
        }
    }

    public function receiveResponseXML(
        string $ticket,
        string $response,
        string $hresult,
        string $message
    ): int {
        // Save the last 'processing' row; mark done/error; return percent (0-100)
        try {
            $pdo = pdo();
            $row = $pdo->query("
                SELECT * FROM " . QUEUE_TABLE . "
                 WHERE status='processing'
                 ORDER BY qbxml_sent_at DESC, id DESC
                 LIMIT 1
            ")->fetch();

            if ($row) {
                $status = ($hresult === '' ? 'done' : 'error');
                $stmt = $pdo->prepare("
                    UPDATE " . QUEUE_TABLE . "
                       SET status=:status,
                           qbxml_response=:resp,
                           last_error=:err,
                           updated_at=NOW()
                     WHERE id=:id
                ");
                $stmt->execute([
                    ':status' => $status,
                    ':resp'   => $response,
                    ':err'    => ($hresult === '' ? null : ($hresult . ' ' . $message)),
                    ':id'     => $row['id'],
                ]);
                log_line("receiveResponseXML -> id={$row['id']} status=$status, hresult='$hresult'");
            } else {
                log_line("receiveResponseXML -> no matching 'processing' row");
            }
        } catch (Throwable $e) {
            log_line("receiveResponseXML error: " . $e->getMessage());
        }

        // 100 tells QBWC we’re done for this update cycle
        return 100;
    }

    public function getLastError(string $ticket): string {
        try {
            $pdo = pdo();
            $msg = $pdo->query("
                SELECT COALESCE(last_error,'') AS e
                  FROM " . QUEUE_TABLE . "
                 WHERE status='error'
                 ORDER BY updated_at DESC, id DESC
                 LIMIT 1
            ")->fetchColumn();
            $msg = (string)$msg;
            log_line("getLastError -> " . ($msg === '' ? '<empty>' : $msg));
            return $msg;
        } catch (Throwable $e) {
            log_line("getLastError error: " . $e->getMessage());
            return 'Server error.';
        }
    }

    public function connectionError(string $ticket, string $hresult, string $message): string {
        log_line("connectionError hresult=$hresult message=$message");
        return 'done';
    }

    public function closeConnection(string $ticket): string {
        log_line("closeConnection(ticket=$ticket) -> OK");
        return 'OK';
    }
}

/*───────────────────────────────────────────────────────────────────────────*/
/* SOAP Server                                                               */
/*───────────────────────────────────────────────────────────────────────────*/
/* ── SOAP Server ───────────────────────────────────────────────────────── */

ini_set('soap.wsdl_cache_enabled', '0');
ini_set('soap.wsdl_cache_ttl', '0');

$WSDL_PATH = __DIR__ . '/qbwc.wsdl';

/* serve the WSDL locally */
if (isset($_GET['wsdl'])) {
    header('Content-Type: text/xml; charset=UTF-8');
    readfile($WSDL_PATH);
    exit;
}

/* make sure no output is sent before/around SOAP */
while (ob_get_level() > 0) { @ob_end_clean(); }

try {
    $server = new SoapServer($WSDL_PATH, [
        'soap_version' => SOAP_1_1,
        'encoding'     => 'UTF-8',
        'cache_wsdl'   => WSDL_CACHE_NONE,
        // 'trace'     => 1, // enable only if you need wire dumps
    ]);

    // IMPORTANT: bind a real instance
    // If your constructor needs $pdo (or other deps), pass it here.
    $svc = new QBWCService($pdo);          // or: new QBWCService();

    // quick safety check — logs helpful if a method name is wrong/not public
    if (!method_exists($svc, 'authenticate') ||
        !method_exists($svc, 'clientVersion') ||
        !method_exists($svc, 'serverVersion')) {
        log_line('Diag: one or more SOAP methods not found on QBWCService');
    }

    $server->setObject($svc);              // <— bind the instance
    $server->handle();

} catch (Throwable $e) {
    log_line('FATAL SOAP error: ' . $e->getMessage());
    header('Content-Type: text/xml; charset=utf-8');
    echo '<?xml version="1.0" encoding="UTF-8"?>'
       . '<SOAP-ENV:Envelope xmlns:SOAP-ENV="http://schemas.xmlsoap.org/soap/envelope/">'
       . '<SOAP-ENV:Body><SOAP-ENV:Fault><faultcode>SOAP-ENV:Server</faultcode>'
       . '<faultstring>Internal error</faultstring></SOAP-ENV:Fault></SOAP-ENV:Body>'
       . '</SOAP-ENV:Envelope>';
}