<?php
/**
 * QuickBooks Web Connector bridge for OpenCart
 * Fully patched: stops infinite loop, wraps valid QBXML, adds item if missing,
 * then adds customer and invoice. Uses PHP sessions keyed by QBWC ticket.
 *
 * PHP 7.4+
 */

declare(strict_types=1);

// ======================== CONFIG ========================
const BRIDGE_VERSION            = '2.1.0';            // serverVersion()
const COMPANY_FILE              = 'C:\Users\Public\Documents\Intuit\QuickBooks\Company Files\testintegration.qbw';

// Defaults used when building minimal artifacts for a smoke test
const QB_DEFAULT_ITEM_FULLNAME  = 'OC Web Item';      // Non-inventory item we ensure exists
const QB_INCOME_ACCOUNT_NAME    = 'Sales';            // Must exist in QB (Income)
const QB_TAX_CODE_NAME          = 'Non';              // Typical non-tax code; change if you use tax
const QB_CUSTOMER_FULLNAME      = 'Web Customer';     // Simple catch-all customer for test invoices

// Invoice test values (used if your integration doesn't inject a real order)
const TEST_INVOICE_REFNUMBER    = 'WEB-TEST-001';
const TEST_TXN_DATE             = null;               // null => today (QB format YYYY-MM-DD)
const TEST_INVOICE_AMOUNT       = 1.00;               // One-dollar test line

// Log file (ensure web user can write here)
const LOG_FILE                  = __DIR__ . '/qbwc-bridge.log';

// Reply with this QBXML version in the PI
const QBXML_VERSION_PI          = '16.0';

// ========================================================

/** Simple logger (never throws) */
function log_line(string $msg): void
{
    $stamp = gmdate('Y-m-d H:i:s') . ' UTC';
    // Avoid log growth without bound (rotate roughly at ~5 MB)
    try {
        if (file_exists(LOG_FILE) && filesize(LOG_FILE) > 5_000_000) {
            @rename(LOG_FILE, LOG_FILE . '.' . date('Ymd_His'));
        }
        @file_put_contents(LOG_FILE, "[$stamp] $msg\n", FILE_APPEND);
    } catch (\Throwable $e) {
        // ignore
    }
}

/** Escape for XML text nodes */
function x(string $s): string
{
    return htmlspecialchars($s, ENT_XML1 | ENT_COMPAT, 'UTF-8');
}

/** Today as QB date (YYYY-MM-DD) */
function qb_date(?string $iso = null): string
{
    if ($iso) return substr($iso, 0, 10);
    return date('Y-m-d');
}

/** Wrap inner QBXML requests in a valid envelope */
function qbxml_envelope(string $innerRq, string $onError = 'continueOnError'): string
{
    $xml = '<?xml version="1.0" encoding="utf-8"?>'
         . '<?qbxml version="' . QBXML_VERSION_PI . '"?>'
         . '<QBXML><QBXMLMsgsRq onError="' . x($onError) . '">'
         . $innerRq
         . '</QBXMLMsgsRq></QBXML>';

    return $xml;
}

/** Build ItemNonInventoryQueryRq by FullName */
function qbxml_item_query(string $fullName): string
{
    $rq = '<ItemNonInventoryQueryRq>'
        . '<FullName>' . x($fullName) . '</FullName>'
        . '</ItemNonInventoryQueryRq>';
    return qbxml_envelope($rq, 'stopOnError');
}

/** Build ItemNonInventoryAddRq */
function qbxml_item_add(string $fullName, string $incomeAccountName, string $taxCodeName): string
{
    $rq = '<ItemNonInventoryAddRq>'
        . '<ItemNonInventoryAdd>'
        . '<Name>' . x($fullName) . '</Name>'
        . '<IncomeAccountRef><FullName>' . x($incomeAccountName) . '</FullName></IncomeAccountRef>'
        . '<SalesTaxCodeRef><FullName>' . x($taxCodeName) . '</FullName></SalesTaxCodeRef>'
        . '</ItemNonInventoryAdd>'
        . '</ItemNonInventoryAddRq>';
    return qbxml_envelope($rq);
}

/** Build CustomerAddRq (minimal) */
function qbxml_customer_add(string $fullName): string
{
    $rq = '<CustomerAddRq>'
        . '<CustomerAdd>'
        . '<Name>' . x($fullName) . '</Name>'
        . '</CustomerAdd>'
        . '</CustomerAddRq>';
    return qbxml_envelope($rq);
}

/** Build InvoiceAddRq with a single item line */
function qbxml_invoice_add(
    string $customerFullName,
    string $itemFullName,
    float $amount,
    ?string $refNumber = null,
    ?string $txnDateISO = null
): string {
    $ref   = $refNumber ? '<RefNumber>' . x($refNumber) . '</RefNumber>' : '';
    $date  = '<TxnDate>' . x(qb_date($txnDateISO)) . '</TxnDate>';
    // Non-tax, single line
    $line  = '<InvoiceLineAdd>'
           . '<ItemRef><FullName>' . x($itemFullName) . '</FullName></ItemRef>'
           . '<Desc>' . x($itemFullName) . '</Desc>'
           . '<Quantity>1</Quantity>'
           . '<Rate>' . number_format($amount, 2, '.', '') . '</Rate>'
           . '</InvoiceLineAdd>';

    $rq =
        '<InvoiceAddRq>'
      . '<InvoiceAdd>'
      . '<CustomerRef><FullName>' . x($customerFullName) . '</FullName></CustomerRef>'
      . $ref
      . $date
      . $line
      . '</InvoiceAdd>'
      . '</InvoiceAddRq>';

    return qbxml_envelope($rq);
}

/** State helpers (kept per QBWC ticket in PHP session) */
function start_ticket_session(string $ticket): void
{
    // Use ticket as session id (safe chars)
    if (!preg_match('/^[A-Za-z0-9_\-]+$/', $ticket)) {
        $ticket = preg_replace('/[^A-Za-z0-9_\-]/', '_', $ticket);
    }
    if (session_status() === PHP_SESSION_ACTIVE) {
        session_write_close();
    }
    session_id('QBWC_' . $ticket);
    session_start([
        'read_and_close' => false,
        'cookie_httponly' => true,
        'cookie_secure' => isset($_SERVER['HTTPS']),
        'use_cookies' => false,   // QBWC is not browser; avoid cookies
        'use_only_cookies' => false,
        'use_strict_mode' => false,
        'gc_maxlifetime' => 3600
    ]);

    if (!isset($_SESSION['flow'])) {
        $_SESSION['flow'] = [
            'queue' => [],     // array of ['type' => one of ensure_item_query|ensure_item_add|customer_add|invoice_add, 'xml' => '...']
            'cursor' => 0,
            'vars' => [        // scratch area
                'item_found' => false,
                'item_listid' => null,
                'customer_listid' => null,
                'last_error' => '',
            ],
            'bootstrapped' => false,
        ];
    }
}

/** Build the session queue (only once per ticket) */
function ensure_flow_bootstrapped(): void
{
    if ($_SESSION['flow']['bootstrapped']) return;

    // 1) Ensure item exists
    $_SESSION['flow']['queue'][] = ['type' => 'ensure_item_query', 'xml' => qbxml_item_query(QB_DEFAULT_ITEM_FULLNAME)];
    // If not found, we will enqueue 'ensure_item_add' dynamically in receiveResponseXML()

    // 2) Customer add (minimal). Using continueOnError (envelope default) lets QB accept if duplicate.
    $_SESSION['flow']['queue'][] = ['type' => 'customer_add', 'xml' => qbxml_customer_add(QB_CUSTOMER_FULLNAME)];

    // 3) Invoice add (single line using the default item)
    $_SESSION['flow']['queue'][] = [
        'type' => 'invoice_add',
        'xml'  => qbxml_invoice_add(
            QB_CUSTOMER_FULLNAME,
            QB_DEFAULT_ITEM_FULLNAME,
            TEST_INVOICE_AMOUNT,
            TEST_INVOICE_REFNUMBER,
            TEST_TXN_DATE
        )
    ];

    $_SESSION['flow']['bootstrapped'] = true;
}

/** Return next XML or empty string if done */
function next_request_xml(): string
{
    $q = &$_SESSION['flow']['queue'];
    $i = &$_SESSION['flow']['cursor'];

    if ($i >= count($q)) {
        return '';
    }
    $xml = (string) $q[$i]['xml'];
    $safePreview = substr($xml, 0, 300);
    log_line("sendRequestXML: step " . ($i + 1) . "/" . count($q) . " type={$q[$i]['type']} bytes=" . strlen($xml) . " preview=" . str_replace(["\r","\n"], '', $safePreview) . '...');
    return $xml;
}

/** Advance cursor to next request */
function advance_cursor(): void
{
    $_SESSION['flow']['cursor'] = (int)$_SESSION['flow']['cursor'] + 1;
}

/** Handle QuickBooks responses and decide the next step percentage */
function handle_response_and_progress(string $response, ?string $hresult, ?string $message): int
{
    // If QuickBooks signals a parse/processing error, stop immediately (prevents infinite loop)
    if (!empty($hresult) || !empty($message)) {
        $_SESSION['flow']['vars']['last_error'] = trim(($hresult ?? '') . ' ' . ($message ?? ''));
        log_line("receiveResponseXML: HALTING due to error hresult={$hresult} message={$message}");
        return 100; // tells QBWC we're done with this run
    }

    $cursor = (int) $_SESSION['flow']['cursor'];
    $current = $_SESSION['flow']['queue'][$cursor] ?? null;
    $type = $current['type'] ?? 'unknown';

    log_line("receiveResponseXML: step " . ($cursor + 1) . " type={$type} bytes=" . strlen($response));

    // Parse XML safely
    $simple = null;
    libxml_use_internal_errors(true);
    try {
        $simple = simplexml_load_string($response);
    } catch (\Throwable $e) {
        $_SESSION['flow']['vars']['last_error'] = 'Failed to parse QBXML response';
        log_line("receiveResponseXML: parse failure => stopping. " . $e->getMessage());
        return 100;
    }
    if (!$simple) {
        $_SESSION['flow']['vars']['last_error'] = 'Invalid QBXML response';
        log_line("receiveResponseXML: invalid QBXML => stopping.");
        return 100;
    }

    // Find the first *Rs node inside QBXMLMsgsRs
    $msgs = $simple->QBXMLMsgsRs ?? null;
    if (!$msgs) {
        log_line("receiveResponseXML: missing QBXMLMsgsRs, continuing but advancing.");
        advance_cursor();
        return progress_percent();
    }

    // Helper: read status on the current *Rs
    $statusCode = null;
    $rsNodeName = null;

    foreach ($msgs->children() as $node) {
        $rsNodeName = $node->getName();
        $statusCode = (string) ($node['statusCode'] ?? '');
        // We only inspect the first node for the current step
        break;
    }

    log_line("receiveResponseXML: Rs={$rsNodeName} statusCode={$statusCode}");

    // Handle per step
    switch ($type) {
        case 'ensure_item_query':
            // If item exists, ItemNonInventoryQueryRs will have ItemNonInventoryRet
            $found = false;
            if (isset($msgs->ItemNonInventoryQueryRs)) {
                $ret = $msgs->ItemNonInventoryQueryRs->ItemNonInventoryRet ?? null;
                if ($ret) {
                    $listid = (string) ($ret->ListID ?? '');
                    if ($listid !== '') {
                        $found = true;
                        $_SESSION['flow']['vars']['item_found'] = true;
                        $_SESSION['flow']['vars']['item_listid'] = $listid;
                        log_line("ensure_item_query: found ListID={$listid}");
                    }
                }
            }
            if (!$found) {
                // Queue an add right after this cursor
                $inject = ['type' => 'ensure_item_add', 'xml' => qbxml_item_add(QB_DEFAULT_ITEM_FULLNAME, QB_INCOME_ACCOUNT_NAME, QB_TAX_CODE_NAME)];
                array_splice($_SESSION['flow']['queue'], $cursor + 1, 0, [$inject]);
                log_line("ensure_item_query: not found => injecting ensure_item_add step");
            }
            advance_cursor();
            break;

        case 'ensure_item_add':
            // Expect ItemNonInventoryAddRs with statusCode 0 (success) or 3100 (already exists) if name collision
            if ($statusCode === '0') {
                $ret = $msgs->ItemNonInventoryAddRs->ItemNonInventoryRet ?? null;
                $listid = $ret ? (string) ($ret->ListID ?? '') : '';
                $_SESSION['flow']['vars']['item_found'] = true;
                $_SESSION['flow']['vars']['item_listid'] = $listid ?: $_SESSION['flow']['vars']['item_listid'];
                log_line("ensure_item_add: added OK ListID=" . ($_SESSION['flow']['vars']['item_listid'] ?? ''));
            } elseif ($statusCode === '3100') {
                // Name already exists – treat as found
                $_SESSION['flow']['vars']['item_found'] = true;
                log_line("ensure_item_add: status 3100 (already exists) => proceed");
            } else {
                $_SESSION['flow']['vars']['last_error'] = "Item add failed (status {$statusCode})";
                log_line("ensure_item_add: unexpected status {$statusCode} => stopping after advancing");
            }
            advance_cursor();
            break;

        case 'customer_add':
            // status 0 success; 3100 duplicate; both are fine
            if ($statusCode === '0' || $statusCode === '3100') {
                log_line("customer_add: status {$statusCode} OK/dup => proceed");
                advance_cursor();
            } else {
                $_SESSION['flow']['vars']['last_error'] = "Customer add failed (status {$statusCode})";
                log_line("customer_add: status {$statusCode} => proceed anyway to avoid stalling");
                advance_cursor(); // Do not stall the queue
            }
            break;

        case 'invoice_add':
            if ($statusCode === '0') {
                log_line("invoice_add: success");
            } else {
                log_line("invoice_add: status {$statusCode}");
            }
            advance_cursor();
            break;

        default:
            log_line("receiveResponseXML: unrecognized step '{$type}', advancing");
            advance_cursor();
            break;
    }

    return progress_percent();
}

/** Compute progress for QBWC (0..100). We’ll map each step equally and finish with 100. */
function progress_percent(): int
{
    $total = max(1, count($_SESSION['flow']['queue']));
    $done  = min($total, (int) $_SESSION['flow']['cursor']);
    $pct   = (int) floor(($done / $total) * 100);
    if ($done >= $total) $pct = 100;
    log_line("progress: {$done}/{$total} => {$pct}%");
    return $pct;
}

/** Minimal WSDL (served at ?wsdl); matches QuickBooks Desktop QBWC method signatures */
function emit_wsdl_and_exit(): void
{
    header('Content-Type: text/xml; charset=utf-8');
    $endpoint = (isset($_SERVER['HTTPS']) ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . $_SERVER['SCRIPT_NAME'];

    // Note: This is a simplified WSDL sufficient for QBWC. If you already have a WSDL, you can keep using it.
    $wsdl = <<<XML
<?xml version="1.0" encoding="UTF-8"?>
<definitions name="QBWebConnectorSvc"
             targetNamespace="http://developer.intuit.com/"
             xmlns:tns="http://developer.intuit.com/"
             xmlns:soap="http://schemas.xmlsoap.org/wsdl/soap/"
             xmlns:xsd="http://www.w3.org/2001/XMLSchema"
             xmlns:wsdl="http://schemas.xmlsoap.org/wsdl/">
  <wsdl:types>
    <xsd:schema targetNamespace="http://developer.intuit.com/">
      <xsd:element name="serverVersion">
        <xsd:complexType/>
      </xsd:element>
      <xsd:element name="serverVersionResponse">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="serverVersionResult" type="xsd:string"/>
          </xsd:sequence>
        </xsd:complexType>
      </xsd:element>

      <xsd:element name="clientVersion">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="strVersion" type="xsd:string"/>
          </xsd:sequence>
        </xsd:complexType>
      </xsd:element>
      <xsd:element name="clientVersionResponse">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="clientVersionResult" type="xsd:string"/>
          </xsd:sequence>
        </xsd:complexType>
      </xsd:element>

      <xsd:element name="authenticate">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="strUserName" type="xsd:string"/>
            <xsd:element name="strPassword" type="xsd:string"/>
          </xsd:sequence>
        </xsd:complexType>
      </xsd:element>
      <xsd:element name="authenticateResponse">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="authenticateResult" type="xsd:ArrayOfString" xmlns:xsd="http://schemas.xmlsoap.org/soap/encoding/"/>
          </xsd:sequence>
        </xsd:complexType>
      </xsd:element>

      <xsd:element name="sendRequestXML">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="ticket" type="xsd:string"/>
            <xsd:element name="strHCPResponse" type="xsd:string"/>
            <xsd:element name="strCompanyFileName" type="xsd:string"/>
            <xsd:element name="qbXMLCountry" type="xsd:string"/>
            <xsd:element name="qbXMLMajorVers" type="xsd:string"/>
            <xsd:element name="qbXMLMinorVers" type="xsd:string"/>
          </xsd:sequence>
        </xsd:complexType>
      </xsd:element>
      <xsd:element name="sendRequestXMLResponse">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="sendRequestXMLResult" type="xsd:string"/>
          </xsd:sequence>
        </xsd:complexType>
      </xsd:element>

      <xsd:element name="receiveResponseXML">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="ticket" type="xsd:string"/>
            <xsd:element name="response" type="xsd:string"/>
            <xsd:element name="hresult" type="xsd:string"/>
            <xsd:element name="message" type="xsd:string"/>
          </xsd:sequence>
        </xsd:complexType>
      </xsd:element>
      <xsd:element name="receiveResponseXMLResponse">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="receiveResponseXMLResult" type="xsd:int"/>
          </xsd:sequence>
        </xsd:complexType>
      </xsd:element>

      <xsd:element name="getLastError">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="ticket" type="xsd:string"/>
          </xsd:sequence>
        </xsd:complexType>
      </xsd:element>
      <xsd:element name="getLastErrorResponse">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="getLastErrorResult" type="xsd:string"/>
          </xsd:sequence>
        </xsd:complexType>
      </xsd:element>

      <xsd:element name="closeConnection">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="ticket" type="xsd:string"/>
          </xsd:sequence>
        </xsd:complexType>
      </xsd:element>
      <xsd:element name="closeConnectionResponse">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="closeConnectionResult" type="xsd:string"/>
          </xsd:sequence>
        </xsd:complexType>
      </xsd:element>

    </xsd:schema>
  </wsdl:types>

  <wsdl:message name="serverVersionRequest">
    <wsdl:part name="parameters" element="tns:serverVersion"/>
  </wsdl:message>
  <wsdl:message name="serverVersionResponse">
    <wsdl:part name="parameters" element="tns:serverVersionResponse"/>
  </wsdl:message>

  <wsdl:message name="clientVersionRequest">
    <wsdl:part name="parameters" element="tns:clientVersion"/>
  </wsdl:message>
  <wsdl:message name="clientVersionResponse">
    <wsdl:part name="parameters" element="tns:clientVersionResponse"/>
  </wsdl:message>

  <wsdl:message name="authenticateRequest">
    <wsdl:part name="parameters" element="tns:authenticate"/>
  </wsdl:message>
  <wsdl:message name="authenticateResponse">
    <wsdl:part name="parameters" element="tns:authenticateResponse"/>
  </wsdl:message>

  <wsdl:message name="sendRequestXMLRequest">
    <wsdl:part name="parameters" element="tns:sendRequestXML"/>
  </wsdl:message>
  <wsdl:message name="sendRequestXMLResponse">
    <wsdl:part name="parameters" element="tns:sendRequestXMLResponse"/>
  </wsdl:message>

  <wsdl:message name="receiveResponseXMLRequest">
    <wsdl:part name="parameters" element="tns:receiveResponseXML"/>
  </wsdl:message>
  <wsdl:message name="receiveResponseXMLResponse">
    <wsdl:part name="parameters" element="tns:receiveResponseXMLResponse"/>
  </wsdl:message>

  <wsdl:message name="getLastErrorRequest">
    <wsdl:part name="parameters" element="tns:getLastError"/>
  </wsdl:message>
  <wsdl:message name="getLastErrorResponse">
    <wsdl:part name="parameters" element="tns:getLastErrorResponse"/>
  </wsdl:message>

  <wsdl:message name="closeConnectionRequest">
    <wsdl:part name="parameters" element="tns:closeConnection"/>
  </wsdl:message>
  <wsdl:message name="closeConnectionResponse">
    <wsdl:part name="parameters" element="tns:closeConnectionResponse"/>
  </wsdl:message>

  <wsdl:portType name="QBWebConnectorSvcSoap">
    <wsdl:operation name="serverVersion">
      <wsdl:input message="tns:serverVersionRequest"/>
      <wsdl:output message="tns:serverVersionResponse"/>
    </wsdl:operation>
    <wsdl:operation name="clientVersion">
      <wsdl:input message="tns:clientVersionRequest"/>
      <wsdl:output message="tns:clientVersionResponse"/>
    </wsdl:operation>
    <wsdl:operation name="authenticate">
      <wsdl:input message="tns:authenticateRequest"/>
      <wsdl:output message="tns:authenticateResponse"/>
    </wsdl:operation>
    <wsdl:operation name="sendRequestXML">
      <wsdl:input message="tns:sendRequestXMLRequest"/>
      <wsdl:output message="tns:sendRequestXMLResponse"/>
    </wsdl:operation>
    <wsdl:operation name="receiveResponseXML">
      <wsdl:input message="tns:receiveResponseXMLRequest"/>
      <wsdl:output message="tns:receiveResponseXMLResponse"/>
    </wsdl:operation>
    <wsdl:operation name="getLastError">
      <wsdl:input message="tns:getLastErrorRequest"/>
      <wsdl:output message="tns:getLastErrorResponse"/>
    </wsdl:operation>
    <wsdl:operation name="closeConnection">
      <wsdl:input message="tns:closeConnectionRequest"/>
      <wsdl:output message="tns:closeConnectionResponse"/>
    </wsdl:operation>
  </wsdl:portType>

  <wsdl:binding name="QBWebConnectorSvcSoap" type="tns:QBWebConnectorSvcSoap">
    <soap:binding transport="http://schemas.xmlsoap.org/soap/http" style="document"/>
    <wsdl:operation name="serverVersion">
      <soap:operation soapAction="http://developer.intuit.com/serverVersion"/>
      <wsdl:input><soap:body use="literal"/></wsdl:input>
      <wsdl:output><soap:body use="literal"/></wsdl:output>
    </wsdl:operation>
    <wsdl:operation name="clientVersion">
      <soap:operation soapAction="http://developer.intuit.com/clientVersion"/>
      <wsdl:input><soap:body use="literal"/></wsdl:input>
      <wsdl:output><soap:body use="literal"/></wsdl:output>
    </wsdl:operation>
    <wsdl:operation name="authenticate">
      <soap:operation soapAction="http://developer.intuit.com/authenticate"/>
      <wsdl:input><soap:body use="literal"/></wsdl:input>
      <wsdl:output><soap:body use="literal"/></wsdl:output>
    </wsdl:operation>
    <wsdl:operation name="sendRequestXML">
      <soap:operation soapAction="http://developer.intuit.com/sendRequestXML"/>
      <wsdl:input><soap:body use="literal"/></wsdl:input>
      <wsdl:output><soap:body use="literal"/></wsdl:output>
    </wsdl:operation>
    <wsdl:operation name="receiveResponseXML">
      <soap:operation soapAction="http://developer.intuit.com/receiveResponseXML"/>
      <wsdl:input><soap:body use="literal"/></wsdl:input>
      <wsdl:output><soap:body use="literal"/></wsdl:output>
    </wsdl:operation>
    <wsdl:operation name="getLastError">
      <soap:operation soapAction="http://developer.intuit.com/getLastError"/>
      <wsdl:input><soap:body use="literal"/></wsdl:input>
      <wsdl:output><soap:body use="literal"/></wsdl:output>
    </wsdl:operation>
    <wsdl:operation name="closeConnection">
      <soap:operation soapAction="http://developer.intuit.com/closeConnection"/>
      <wsdl:input><soap:body use="literal"/></wsdl:input>
      <wsdl:output><soap:body use="literal"/></wsdl:output>
    </wsdl:operation>
  </wsdl:binding>

  <wsdl:service name="QBWebConnectorSvc">
    <wsdl:port name="QBWebConnectorSvcSoap" binding="tns:QBWebConnectorSvcSoap">
      <soap:address location="{$endpoint}"/>
    </wsdl:port>
  </wsdl:service>
</definitions>
XML;

    echo $wsdl;
    exit;
}

// ====================== SOAP IMPLEMENTATION ======================

class QBWebConnectorSvc
{
    public function serverVersion(): string
    {
        log_line("SOAP method = serverVersion");
        return BRIDGE_VERSION;
    }

    public function clientVersion(string $strVersion): string
    {
        log_line("SOAP method = clientVersion; client={$strVersion}");
        // Accept all versions; return empty to allow update
        return '';
    }

    public function authenticate(string $strUserName, string $strPassword): array
    {
        // Here you can validate against your system if desired.
        $ticket = 'TOKEN' . substr(sha1($strUserName . '|' . $strPassword . '|' . microtime(true)), 0, 8);
        log_line("SOAP method = authenticate; user={$strUserName} => ticket={$ticket}");

        start_ticket_session($ticket);
        ensure_flow_bootstrapped();

        // Return [ticket, company-file, qbXMLSubscriptionID?, qbXMLCountry?] — we return 4 items like your logs
        return [$ticket, COMPANY_FILE, '', ''];
    }

    public function sendRequestXML(
        string $ticket,
        string $strHCPResponse,
        string $strCompanyFileName,
        string $qbXMLCountry,
        string $qbXMLMajorVers,
        string $qbXMLMinorVers
    ): string {
        log_line("SOAP method = sendRequestXML; ticket={$ticket}");

        start_ticket_session($ticket);
        ensure_flow_bootstrapped();

        $xml = next_request_xml();

        // If queue is empty, return empty string (QBWC will call getLastError then close)
        if ($xml === '') {
            log_line("sendRequestXML: no work => returning empty string");
            return '';
        }
        return $xml;
    }

    public function receiveResponseXML(string $ticket, string $response, string $hresult, string $message): int
    {
        log_line("SOAP method = receiveResponseXML; ticket={$ticket}");
        start_ticket_session($ticket);

        $pct = handle_response_and_progress($response, $hresult, $message);
        return $pct;
    }

    public function getLastError(string $ticket): string
    {
        log_line("SOAP method = getLastError; ticket={$ticket}");
        start_ticket_session($ticket);
        $err = (string) ($_SESSION['flow']['vars']['last_error'] ?? '');
        log_line("getLastError: {$err}");
        return $err;
    }

    public function closeConnection(string $ticket): string
    {
        log_line("SOAP method = closeConnection; ticket={$ticket}");
        // Cleanup the session for this ticket
        start_ticket_session($ticket);
        $_SESSION = [];
        session_destroy();
        return 'OK';
    }
}

// ====================== BOOTSTRAP ROUTER ======================

// Serve a health check
if (isset($_GET['health'])) {
    header('Content-Type: text/plain; charset=utf-8');
    echo "OK " . BRIDGE_VERSION;
    exit;
}

// Serve WSDL if requested
if (isset($_GET['wsdl'])) {
    emit_wsdl_and_exit();
}

// Create SOAP server in *non-WSDL* mode if you prefer; but we will serve our own WSDL as above
$server = null;
if (class_exists('SoapServer')) {
    if (isset($_GET['nowdsl'])) {
        // Non-WSDL mode (fallback)
        $server = new SoapServer(null, ['uri' => 'http://developer.intuit.com/']);
    } else {
        // Use our inline WSDL
        $wsdlUrl = (isset($_SERVER['HTTPS']) ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . $_SERVER['SCRIPT_NAME'] . '?wsdl';
        $server = new SoapServer($wsdlUrl, ['cache_wsdl' => WSDL_CACHE_NONE]);
    }
    $server->setClass(QBWebConnectorSvc::class);

    try {
        $server->handle();
    } catch (\Throwable $e) {
        log_line('SOAP fatal: ' . $e->getMessage());
        // Return a SOAP fault
        $fault = new SoapFault('Server', 'Unhandled exception: ' . $e->getMessage());
        // Echo fault manually if we cannot throw (depends on SAPIs)
        header('Content-Type: text/xml; charset=utf-8');
        echo '<?xml version="1.0" encoding="UTF-8"?>'
           . '<SOAP-ENV:Envelope xmlns:SOAP-ENV="http://schemas.xmlsoap.org/soap/envelope/"><SOAP-ENV:Body>'
           . '<SOAP-ENV:Fault><faultcode>Server</faultcode><faultstring>' . x($fault->getMessage()) . '</faultstring></SOAP-ENV:Fault>'
           . '</SOAP-ENV:Body></SOAP-ENV:Envelope>';
    }
} else {
    header('Content-Type: text/plain; charset=utf-8', true, 500);
    echo "SoapServer extension not available";
}
