<?php
/**
 * OpenCart ↔ QuickBooks Desktop (QBWC) Bridge
 * QB Desktop 2024 / QBWC 34.x
 * Endpoint: qbwc_server.php  |  WSDL: qbwc_server.php?wsdl
 *
 * Behavior:
 * - authenticate() accepts user "qbwc_user"
 * - sendRequestXML() pulls the next OpenCart order with status_id=5 (Completed)
 * - Builds a QBXML batch:
 *     1) ItemNonInventoryAdd for each order line (continueOnError)
 *     2) CustomerAdd for the order's billing name (continueOnError)
 *     3) InvoiceAdd with the order lines (no tax, no shipping)
 */

declare(strict_types=1);
ini_set('display_errors','0');
ini_set('log_errors','1');
ini_set('error_log', __DIR__ . '/qbwc-php-error.log');
ob_start();
date_default_timezone_set('UTC');

// ---- Configuration ---------------------------------------------------------

// Company file path (for display in QBWC only)
const QB_COMPANY_FILE = 'C:\\Users\\Public\\Documents\\Intuit\\QuickBooks\\Company Files\\testintegration.qbw';


// OpenCart DB
const OC_DB_HOST   = 'localhost';
const OC_DB_NAME   = 'rkimport_lat4';
const OC_DB_USER   = 'rkimport_qbws';
const OC_DB_PASS   = '0MWD4.@HgG5F;Dr[';
const OC_DB_PORT   = 3306;
const OC_PREFIX    = 'oc_';

// Order status(es) to sync
const OC_ALLOWED_STATUS_IDS = '5'; // Completed only

// QBXML version
const QBXML_VERSION = '13.0';

// Logging
const LOG_FILE = __DIR__ . '/qbwc-bridge.log';

// ---- Helpers ---------------------------------------------------------------

function log_bridge(string $msg): void {
  file_put_contents(LOG_FILE, sprintf("[%s] %s\n", date('Y-m-d H:i:s'), $msg), FILE_APPEND);
}

function db(): PDO {
  static $pdo = null;
  if ($pdo) return $pdo;
  $dsn = sprintf('mysql:host=%s;port=%d;dbname=%s;charset=utf8mb4', OC_DB_HOST, OC_DB_PORT, OC_DB_NAME);
  $pdo = new PDO($dsn, OC_DB_USER, OC_DB_PASS, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
  return $pdo;
}

function qbwc_state(): array {
  $pdo = db();
  $pdo->exec("CREATE TABLE IF NOT EXISTS qbwc_state (
    id TINYINT PRIMARY KEY,
    last_order_id INT DEFAULT 0,
    last_error TEXT,
    last_run TIMESTAMP NULL DEFAULT NULL,
    ticket VARCHAR(64) DEFAULT NULL,
    progress INT DEFAULT 0
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;");
  $pdo->exec("INSERT IGNORE INTO qbwc_state (id,last_order_id,progress) VALUES (1,0,0)");
  $row = $pdo->query("SELECT * FROM qbwc_state WHERE id=1")->fetch();
  if (!$row) $row = ['id'=>1,'last_order_id'=>0,'last_error'=>null,'last_run'=>null,'ticket'=>null,'progress'=>0];
  return $row;
}
function qbwc_update(array $fields): void {
  $pdo = db();
  $sets=[]; $params=[];
  foreach($fields as $k=>$v){ $sets[]="`$k`=:$k"; $params[":$k"]=$v; }
  $sql = "UPDATE qbwc_state SET ".implode(',', $sets)." WHERE id=1";
  $stmt=$pdo->prepare($sql); $stmt->execute($params);
}

function next_completed_order(int $after): ?array {
  $pdo = db();
  $status = array_filter(array_map('trim', explode(',', OC_ALLOWED_STATUS_IDS)));
  if(!$status) return null;
  $in = implode(',', array_fill(0, count($status), '?'));
  $sql = "SELECT o.order_id, o.firstname, o.lastname, o.email,
                 o.payment_company, o.payment_firstname, o.payment_lastname,
                 o.payment_address_1, o.payment_address_2, o.payment_city,
                 o.payment_postcode, o.payment_zone, o.payment_country,
                 o.total, o.currency_code, o.date_added
          FROM ".OC_PREFIX."order o
          WHERE o.order_status_id IN ($in) AND o.order_id > ?
          ORDER BY o.order_id ASC
          LIMIT 1";
  $stmt = $pdo->prepare($sql);
  $i=1; foreach($status as $s){ $stmt->bindValue($i++,(int)$s,PDO::PARAM_INT); }
  $stmt->bindValue($i,$after,PDO::PARAM_INT);
  $stmt->execute();
  $order = $stmt->fetch();
  if(!$order) return null;
  $stmt2 = $pdo->prepare("SELECT name, model, product_id, quantity, price FROM ".OC_PREFIX."order_product WHERE order_id=?");
  $stmt2->execute([$order['order_id']]);
  $order['items'] = $stmt2->fetchAll();
  return $order;
}

function xml_safe($s){ return htmlspecialchars($s??'', ENT_XML1|ENT_QUOTES); }

function build_qbxml_for_order(array $order): string {
  // Customer full name
  $cust = trim(($order['payment_firstname'] ?? $order['firstname'] ?? '').' '.($order['payment_lastname'] ?? $order['lastname'] ?? ''));
  if($cust==='') $cust = 'Web Customer ' . ($order['email'] ?? '');
  $cust = preg_replace('/\s+/', ' ', $cust);

  // Build ItemNonInventoryAdd list
  $seen = [];
  $itemsAdd = '';
  foreach($order['items'] as $it){
    $full = ($it['model'] ?: $it['name']);
    if(isset($seen[$full])) continue;
    $seen[$full]=true;
    $itemsAdd .= "<ItemNonInventoryAddRq><ItemNonInventoryAdd>"
               . "<Name>".xml_safe(substr($full,0,31))."</Name>"
               . "</ItemNonInventoryAdd></ItemNonInventoryAddRq>";
  }

  // CustomerAdd
  $customerAdd = "<CustomerAddRq><CustomerAdd>"
               . "<Name>".xml_safe(substr($cust,0,41))."</Name>"
               . "<CompanyName>".xml_safe(substr($order['payment_company'] ?: $cust,0,41))."</CompanyName>"
               . "<BillAddress>"
               .   "<Addr1>".xml_safe(substr(($order['payment_company'] ?: $cust),0,41))."</Addr1>"
               .   "<Addr2>".xml_safe(substr(($order['payment_address_1'] ?? ''),0,41))."</Addr2>"
               .   "<Addr3>".xml_safe(substr(($order['payment_address_2'] ?? ''),0,41))."</Addr3>"
               .   "<City>".xml_safe(substr(($order['payment_city'] ?? ''),0,31))."</City>"
               .   "<State>".xml_safe(substr(($order['payment_zone'] ?? ''),0,21))."</State>"
               .   "<PostalCode>".xml_safe(substr(($order['payment_postcode'] ?? ''),0,13))."</PostalCode>"
               .   "<Country>".xml_safe(substr(($order['payment_country'] ?? ''),0,21))."</Country>"
               . "</BillAddress>"
               . "<Email>".xml_safe($order['email'] ?? '')."</Email>"
               . "</CustomerAdd></CustomerAddRq>";

  // Invoice lines
  $lines='';
  foreach($order['items'] as $line){
    $itemName = $line['model'] ?: $line['name'];
    $rate = number_format((float)$line['price'], 2, '.', '');
    $qty  = (int)$line['quantity'];
    $lines .= "<InvoiceLineAdd>"
            .   "<ItemRef><FullName>".xml_safe($itemName)."</FullName></ItemRef>"
            .   "<Desc>".xml_safe($line['name'])."</Desc>"
            .   "<Quantity>{$qty}</Quantity>"
            .   "<Rate>{$rate}</Rate>"
            . "</InvoiceLineAdd>";
  }

  $txnDate = substr($order['date_added'],0,10);
  $invoice = "<InvoiceAddRq requestID=\"".(int)$order['order_id']."\"><InvoiceAdd>"
           . "<CustomerRef><FullName>".xml_safe($cust)."</FullName></CustomerRef>"
           . "<TxnDate>".xml_safe($txnDate)."</TxnDate>"
           . "<PONumber>".xml_safe('OC-'.$order['order_id'])."</PONumber>"
           . "<Memo>".xml_safe('OpenCart Order #'.$order['order_id'])."</Memo>"
           . $lines
           . "</InvoiceAdd></InvoiceAddRq>";

  $qbxml = '<?xml version="1.0" encoding="utf-8"?>'."\n"
         . '<?qbxml version="'.QBXML_VERSION.'"?>'."\n"
         . '<QBXML><QBXMLMsgsRq onError="continueOnError">'
         . $itemsAdd . $customerAdd . $invoice
         . '</QBXMLMsgsRq></QBXML>';
  return $qbxml;
}

// ---- SOAP Service ----------------------------------------------------------
// ---- SOAP Service ----------------------------------------------------------
class QBWCService {

    // Server version: QBWC calls first, empty string is acceptable
    public function serverVersion() {
        return '';
    }

    // Client version: QBWC sends its version, we can always accept
    public function clientVersion($strVersion) {
        return '';
    }

    // Authentication — must return exactly two strings in an array
    // [ticket, companyFilePath]  OR  ['', 'nvu'] if not validated
   public function authenticate($strUserName, $strPassword) {
  log_bridge("AUTH in: user={$strUserName}");
  if ($strUserName !== 'qbwc_user') {
    log_bridge("AUTH nvu for user={$strUserName}");
    return array('', 'nvu'); // exactly 2 strings
  }
  $ticket = bin2hex(random_bytes(8));
  qbwc_update(['ticket'=>$ticket, 'last_run'=>date('Y-m-d H:i:s')]);
  log_bridge("AUTH ok ticket={$ticket}");
  return array($ticket, QB_COMPANY_FILE); // exactly 2 plain strings
}


    // SendRequestXML — QBWC asks “what should I send to QuickBooks?”
    public function sendRequestXML($ticket, $strHCPResponse, $strCompanyFileName, $qbXMLCountry, $qbXMLMajorVers, $qbXMLMinorVers) {
        $s = qbwc_state();
        $order = next_completed_order((int)$s['last_order_id']);
        if (!$order) {
            return ''; // no work
        }

        $xml = build_qbxml_for_order($order);
        qbwc_update(['progress' => (int)$order['order_id']]);

        log_bridge("sendRequestXML for order ".$order['order_id']);
        return $xml;
    }

    // ReceiveResponseXML — QBWC gives us QuickBooks’ response
    public function receiveResponseXML($ticket, $response, $hresult, $message) {
        $s = qbwc_state();
        $processed = (int)($s['progress'] ?: $s['last_order_id']);

        if ($hresult === '') {
            if ($processed > (int)$s['last_order_id']) {
                qbwc_update([
                    'last_order_id' => $processed,
                    'last_error'    => null,
                    'progress'      => 0
                ]);
            }
            return 50; // percentage done
        }

        qbwc_update(['last_error' => "hresult=$hresult message=$message"]);
        return 100; // stop on error
    }

    // If QuickBooks cannot connect
    public function connectionError($ticket, $hresult, $message) {
        log_bridge("connectionError: hresult=$hresult message=$message");
        return 'done';
    }

    // Last error report
    public function getLastError($ticket) {
        $s = qbwc_state();
        return $s['last_error'] ?? '';
    }

    // Close connection after session
    public function closeConnection($ticket) {
        return 'OK';
    }
}

// ---- Front Controller ------------------------------------------------------
// ---- Front Controller ------------------------------------------------------
try {
  // Build self URL
  $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
  $self   = $scheme.'://'.$_SERVER['HTTP_HOST'].$_SERVER['SCRIPT_NAME'];
  $qs     = strtolower($_SERVER['QUERY_STRING'] ?? '');

  // 1) Serve the WSDL for QBWC to download/inspect
  if ($qs === 'wsdl') {
    if (ob_get_length()) ob_clean();
    header('Content-Type: text/xml; charset=utf-8');
    // IMPORTANT: qbwc.wsdl must be UTF-8 **without BOM**
    readfile(__DIR__.'/qbwc.wsdl');
    exit;
  }

  // 2) Runtime SOAP handling
 // 2) RUNTIME SOAP HANDLING — FORCE NON-WSDL MODE
// 2) RUNTIME SOAP HANDLING — FORCE NON-WSDL MODE
ini_set('soap.wsdl_cache_enabled', '0');
ini_set('soap.wsdl_cache_ttl', '0');
if (function_exists('opcache_reset')) { @opcache_reset(); } // nuke opcode cache

$server = new SoapServer(null, [
  'uri'          => 'http://developer.intuit.com/',
  'encoding'     => 'UTF-8',
  'features'     => SOAP_SINGLE_ELEMENT_ARRAYS,
  'soap_version' => SOAP_1_1,
]);

$server->setClass('QBWCService');
if (ob_get_length()) ob_clean();
$server->handle();



} catch (Throwable $e) {
  log_bridge('FATAL: '.$e->getMessage());
  if (ob_get_length()) ob_clean();
  header('Content-Type: text/xml; charset=utf-8');
  echo '<?xml version="1.0" encoding="utf-8"?>'.
       '<SOAP-ENV:Envelope xmlns:SOAP-ENV="http://schemas.xmlsoap.org/soap/envelope/">'.
       '<SOAP-ENV:Body><SOAP-ENV:Fault><faultcode>Server</faultcode>'.
       '<faultstring>Internal error</faultstring></SOAP-ENV:Fault></SOAP-ENV:Body></SOAP-ENV:Envelope>';
  exit;
}

